#include "main include file.h"
#include "D3DSkinnedMesh.h"
#include "MS3DFile.h"
#include "d3dutil.h"
#include <shlwapi.h>
#include <set>
#include <float.h> // for FLT_MAX only
/*#include "d3dxex.h"*/


CD3DSkinnedMesh::CD3DSkinnedMesh()
:	m_pMS3DFile(NULL),
	m_pDevice(NULL),
	m_pVertexBuffer(NULL),
	m_pEdgeBuffer(NULL),
	m_pBoneBuffer(NULL),
	m_bDefaultShaderOnly(FALSE),
	m_bHasFlatNormals(FALSE),
	m_fTime(-1.0f),
	m_fLowestTime(0.0f)
{
	// initialize default shader
	ZeroMemory(&m_shaderDefault, sizeof(D3DShader_t));
	m_shaderDefault.mat.Ambient.r = 0.2f;
	m_shaderDefault.mat.Ambient.g = 0.2f;
	m_shaderDefault.mat.Ambient.b = 0.2f;
	m_shaderDefault.mat.Ambient.a = 1.0f;
	m_shaderDefault.mat.Diffuse.r = 0.8f;
	m_shaderDefault.mat.Diffuse.g = 0.8f;
	m_shaderDefault.mat.Diffuse.b = 0.8f;
	m_shaderDefault.mat.Diffuse.a = 1.8f;
	m_shaderDefault.fTransparency = 1.0f;
}

CD3DSkinnedMesh::~CD3DSkinnedMesh()
{
//	_ASSERTE(m_pDevice == NULL);
}

HRESULT CD3DSkinnedMesh::CreateFromMS3DFile(LPDIRECT3DDEVICE8 pDevice, CMS3DFile *pMS3DFile, LPCTSTR lpszModelPath)
{
	if (!pDevice)
	{
//		_RPT0(_CRT_WARN, "Invalid pointer LPDIRECT3DDEVICE8 pDevice.\n");
		return E_INVALIDARG;
	}

	if (!pMS3DFile)
	{
	//	_RPT0(_CRT_WARN, "Invalid pointer CMS3DFile *pMS3DFile.\n");
		return E_INVALIDARG;
	}


	HRESULT hr;
	hr = Destroy();
	if (FAILED(hr))
	{
//		_RPT0(_CRT_WARN, "Destroy() failed in CD3DSkinnedMesh::CreateFromMS3DFile().\n");
		return E_FAIL;
	}

	DWORD dwTime = ::GetTickCount();

	m_pMS3DFile = pMS3DFile;
	m_pDevice = pDevice;

	DWORD dwNumFaces = pMS3DFile->GetNumTriangles();
	DWORD dwNumVertices = dwNumFaces * 3;
	DWORD dwVertexSize = D3DXGetFVFVertexSize(D3DFVF_VERTEX);

	m_arrVertices.resize(dwNumVertices);
	m_arrVertexBoneIds.resize(dwNumVertices);
	m_arrFlatNormals.resize(dwNumFaces);
	m_arrSubsets.resize(pMS3DFile->GetNumGroups());

	dwNumVertices = 0;
	dwNumFaces = 0;

	DWORD i, j, k;

	//
	// subsets and triangles
	//
	for (i = 0; i < pMS3DFile->GetNumGroups(); i++)
	{
		ms3d_group_t *pGroup = NULL;
		pMS3DFile->GetGroupAt(i, &pGroup);

		DWORD dwAttribId = pGroup->materialIndex + 1;
		m_arrSubsets[i].AttribId = dwAttribId;
		m_arrSubsets[i].VertexStart = dwNumVertices;

		for (j = 0; j < pGroup->numtriangles; j++)
		{
			ms3d_triangle_t *pTriangle = NULL;
			pMS3DFile->GetTriangleAt(pGroup->triangleIndices[j], &pTriangle);

			for (k = 0; k < 3; k++)
			{
				int nVertexIndex = pTriangle->vertexIndices[k];
				ms3d_vertex_t *pVertex = NULL;
				pMS3DFile->GetVertexAt(nVertexIndex, &pVertex);

				m_arrVertices[dwNumVertices].p[0] = pVertex->vertex[0];
				m_arrVertices[dwNumVertices].p[1] = pVertex->vertex[1];
				m_arrVertices[dwNumVertices].p[2] = pVertex->vertex[2];
				m_arrVertices[dwNumVertices].n[0] = pTriangle->vertexNormals[k][0];
				m_arrVertices[dwNumVertices].n[1] = pTriangle->vertexNormals[k][1];
				m_arrVertices[dwNumVertices].n[2] = pTriangle->vertexNormals[k][2];
				m_arrVertices[dwNumVertices].uv[0] = pTriangle->s[k];
				m_arrVertices[dwNumVertices].uv[1] = pTriangle->t[k];
				m_arrVertexBoneIds[dwNumVertices] = pVertex->boneId;
				++dwNumVertices;
			}

			// calc and keep a copy of the face normal
			D3DXVECTOR3 v1 = m_arrVertices[dwNumVertices - 2].p - m_arrVertices[dwNumVertices - 1].p;
			D3DXVECTOR3 v2 = m_arrVertices[dwNumVertices - 2].p - m_arrVertices[dwNumVertices - 3].p;
			D3DXVECTOR3 vFaceNormal;
			D3DXVec3Cross(&vFaceNormal, &v1, &v2);
			D3DXVec3Normalize(&m_arrFlatNormals[dwNumFaces], &vFaceNormal);
			++dwNumFaces;
		}

		m_arrSubsets[i].VertexCount = dwNumVertices - m_arrSubsets[i].VertexStart;
	}

//	_ASSERTE(dwNumVertices == m_arrVertices.size());
	//_ASSERTE(dwNumFaces == pMS3DFile->GetNumTriangles());

	m_arrRenderVertices.resize(dwNumVertices);

//	_ASSERTE(m_pVertexBuffer == NULL);
	hr = m_pDevice->CreateVertexBuffer(dwVertexSize * dwNumVertices, D3DUSAGE_DYNAMIC|D3DUSAGE_WRITEONLY,D3DFVF_VERTEX, D3DPOOL_DEFAULT, &m_pVertexBuffer);

	if (FAILED(hr))
	{
//		_RPT0(_CRT_WARN, "CreateVertexBuffer() failed in CD3DSkinnedMesh::CreateFromMS3DFile().\n");
		return E_FAIL;
	}

//	_ASSERTE(m_pEdgeBuffer == NULL);
	DWORD dwNumEdges = pMS3DFile->GetNumEdges();
	dwVertexSize = D3DXGetFVFVertexSize(D3DFVF_XYZ);
	hr = m_pDevice->CreateVertexBuffer(dwVertexSize * dwNumEdges * 2, D3DUSAGE_DYNAMIC|D3DUSAGE_WRITEONLY,
		D3DFVF_XYZ, D3DPOOL_DEFAULT, &m_pEdgeBuffer);

	if (FAILED(hr))
	{
//		_RPT0(_CRT_WARN, "CreateVertexBuffer() failed in CD3DSkinnedMesh::CreateFromMS3DFile().\n");
		return E_FAIL;
	}

//	_ASSERTE(m_pBoneBuffer == NULL);
	DWORD dwNumBones = pMS3DFile->GetNumJoints();
	if (dwNumBones > 0)
	{
		dwVertexSize = D3DXGetFVFVertexSize(D3DFVF_XYZ);
		hr = m_pDevice->CreateVertexBuffer(dwVertexSize * dwNumBones * 2, D3DUSAGE_DYNAMIC|D3DUSAGE_WRITEONLY,
			D3DFVF_XYZ, D3DPOOL_DEFAULT, &m_pBoneBuffer);

		if (FAILED(hr))
		{
//			_RPT0(_CRT_WARN, "CreateVertexBuffer() failed in CD3DSkinnedMesh::CreateFromMS3DFile().\n");
			return E_FAIL;
		}
	}

	SetupBones();
	SetTime(m_fTime);
	UpdateVertexBuffer();

	//
	// shaders
	//
	m_arrShaders.resize(pMS3DFile->GetNumMaterials() + 1);
	m_arrShaders[0] = m_shaderDefault;
	for (i = 0; i < pMS3DFile->GetNumMaterials(); i++)
	{
		DWORD dwShaderIndex = i + 1;
		ms3d_material_t *pMaterial = NULL;
		pMS3DFile->GetMaterialAt(i, &pMaterial);
		m_arrShaders[dwShaderIndex].mat.Ambient.r = pMaterial->ambient[0];
		m_arrShaders[dwShaderIndex].mat.Ambient.g = pMaterial->ambient[1];
		m_arrShaders[dwShaderIndex].mat.Ambient.b = pMaterial->ambient[2];
		m_arrShaders[dwShaderIndex].mat.Ambient.a = pMaterial->transparency;//pMaterial->ambient[3];
		m_arrShaders[dwShaderIndex].mat.Diffuse.r = pMaterial->diffuse[0];
		m_arrShaders[dwShaderIndex].mat.Diffuse.g = pMaterial->diffuse[1];
		m_arrShaders[dwShaderIndex].mat.Diffuse.b = pMaterial->diffuse[2];
		m_arrShaders[dwShaderIndex].mat.Diffuse.a = pMaterial->transparency;//pMaterial->diffuse[3];
		m_arrShaders[dwShaderIndex].mat.Specular.r = pMaterial->specular[0];
		m_arrShaders[dwShaderIndex].mat.Specular.g = pMaterial->specular[1];
		m_arrShaders[dwShaderIndex].mat.Specular.b = pMaterial->specular[2];
		m_arrShaders[dwShaderIndex].mat.Specular.a = pMaterial->transparency;//pMaterial->specular[3];
		m_arrShaders[dwShaderIndex].mat.Emissive.r = pMaterial->emissive[0];
		m_arrShaders[dwShaderIndex].mat.Emissive.g = pMaterial->emissive[1];
		m_arrShaders[dwShaderIndex].mat.Emissive.b = pMaterial->emissive[2];
		m_arrShaders[dwShaderIndex].mat.Emissive.a = pMaterial->transparency;//pMaterial->emissive[3];
		m_arrShaders[dwShaderIndex].mat.Power = pMaterial->shininess;
		m_arrShaders[dwShaderIndex].bSphereMap = pMaterial->mode & 0x80;
		m_arrShaders[dwShaderIndex].fTransparency = pMaterial->transparency;

		m_arrShaders[dwShaderIndex].pTexture = NULL;


		char pathTemp[PATH_MAX+1];
		

		if (strlen(pMaterial->texture) > 0)
		{

			char szPath[MAX_PATH];
			
			//Detemrine length of path:
			int pathLength;
			for ( pathLength = strlen( lpszModelPath ); --pathLength; ) {
				if ( lpszModelPath[pathLength] == '/' || lpszModelPath[pathLength] == '\\' ) {
					break;
				}
			}



			strncpy( pathTemp, lpszModelPath, pathLength );
			
			if ( pathLength > 0 ) {
				pathTemp[pathLength++] = '/';
			}

			//Check for relative path.
			if ( strncmp( pMaterial->texture, ".\\", 2 ) == 0 ) {
				char pathTemp[PATH_MAX+1];
				// MS3D 1.5.x relative path
				strcpy( pathTemp, lpszModelPath ); 
				strcpy( pathTemp + pathLength, pMaterial->texture + 2 );
				lstrcpy(szPath, pathTemp);
			}
			else {
				// MS3D 1.4.x or earlier - absolute path
			
			//It's nice if we remove the directory..
				int dirLength;
				for ( dirLength = strlen( pMaterial->texture ); --dirLength; ) {
					if (  pMaterial->texture[dirLength] == '/' ||  pMaterial->texture[dirLength] == '\\' ) {
						break;
					}
				}
				
				//Only copy filename!
				strcpy( pathTemp, lpszModelPath ); 
				strcpy( pathTemp + pathLength, pMaterial->texture+dirLength+1);
				lstrcpy(szPath, pathTemp);
			
				hr = D3DXCreateTextureFromFile(m_pDevice, szPath, &m_arrShaders[dwShaderIndex].pTexture); 
			}
				//if (FAILED(hr))
///				DXTRACE_ERR(DXGetErrorDescription9(hr), hr);
		}
	}

	DWORD dwDiffTime = ::GetTickCount() - dwTime;
//	_RPT1(_CRT_WARN, "CD3DSkinnedMesh::CreateFromMS3DFile() used %d ms.\n", dwDiffTime);

	m_fTotalFrames = pMS3DFile->GetTotalFrames();

	return S_OK;
}

HRESULT CD3DSkinnedMesh::UpdateVertexBuffer()
{
	if (m_pVertexBuffer && m_pEdgeBuffer && m_pMS3DFile)
	{
		DWORD i;

		if (m_fTime < 0.0f)
		{
			// update render vertices
			memcpy(&m_arrRenderVertices[0], &m_arrVertices[0], sizeof(D3DVertex_t) * m_arrVertices.size());
			if (m_bHasFlatNormals)
			{
				for (i = 0; i < m_arrRenderVertices.size(); i++)
					m_arrRenderVertices[i].n = m_arrFlatNormals[i / 3];
			}
		}

		// animate here
		else
		{
			for (i = 0; i < m_arrVertices.size(); i++)
			{
				int nBoneIndex = m_arrVertexBoneIds[i];
				if (nBoneIndex >= 0 && nBoneIndex < (int) m_arrBones.size())
				{
					D3DXVECTOR3 v;
					D3DXVec3TransformCoord(&v, &m_arrVertices[i].p, &m_arrBones[nBoneIndex].matWorldInv);
					D3DXVec3TransformCoord(&m_arrRenderVertices[i].p, &v, &m_arrBones[nBoneIndex].matWorldAnim);

					D3DXVECTOR3 n;
					if (m_bHasFlatNormals)
						D3DXVec3TransformNormal(&n, &m_arrFlatNormals[i / 3], &m_arrBones[nBoneIndex].matWorldInv);
					else
						D3DXVec3TransformNormal(&n, &m_arrVertices[i].n, &m_arrBones[nBoneIndex].matWorldInv);
						D3DXVec3TransformNormal(&m_arrRenderVertices[i].n, &n, &m_arrBones[nBoneIndex].matWorldAnim);
				}
				else
				{
					memcpy(&m_arrRenderVertices[i], &m_arrVertices[i], sizeof(D3DVertex_t));
					if (m_bHasFlatNormals)
						m_arrRenderVertices[i].n = m_arrFlatNormals[i / 3];
				}
			}
		}

		// copy content of m_arrRenderVertices to vertex buffer
		HRESULT hr;
		D3DVertex_t *pVertices = NULL;
		hr = m_pVertexBuffer->Lock(0, 0, (BYTE **) &pVertices, D3DLOCK_DISCARD);
		if (SUCCEEDED(hr))
		{
			memcpy(pVertices, &m_arrRenderVertices[0], sizeof(D3DVertex_t) * m_arrRenderVertices.size());
			hr = m_pVertexBuffer->Unlock();

		}

		D3DXVECTOR3 *pEdgeVertices = NULL;
		hr = m_pEdgeBuffer->Lock(0, 0, (BYTE **) &pEdgeVertices, D3DLOCK_DISCARD);
		if (SUCCEEDED(hr))
		{
			for (i = 0; i < m_pMS3DFile->GetNumEdges(); i++)
			{
				ms3d_edge_t *pEdge = NULL;
				m_pMS3DFile->GetEdgeAt(i, &pEdge);

				ms3d_vertex_t *pVertex1 = NULL;
				m_pMS3DFile->GetVertexAt(pEdge->edgeIndices[0], &pVertex1);
				ms3d_vertex_t *pVertex2 = NULL;
				m_pMS3DFile->GetVertexAt(pEdge->edgeIndices[1], &pVertex2);

				D3DXVECTOR3 v;

				v.x = pVertex1->vertex[0];
				v.y = pVertex1->vertex[1];
				v.z = pVertex1->vertex[2];
				int nBoneIndex1 = pVertex1->boneId;
				if (nBoneIndex1 >= 0 && nBoneIndex1 < (int) m_arrBones.size())
				{
					D3DXVec3TransformCoord(&v, &v, &m_arrBones[nBoneIndex1].matWorldInv);
					D3DXVec3TransformCoord(&pEdgeVertices[i * 2 + 0], &v, &m_arrBones[nBoneIndex1].matWorldAnim);
				}
				else
				{
					pEdgeVertices[i * 2 + 0] = v;
				}

				v.x = pVertex2->vertex[0];
				v.y = pVertex2->vertex[1];
				v.z = pVertex2->vertex[2];
				int nBoneIndex2 = pVertex2->boneId;
				if (nBoneIndex2 >= 0 && nBoneIndex2 < (int) m_arrBones.size())
				{
					D3DXVec3TransformCoord(&v, &v, &m_arrBones[nBoneIndex2].matWorldInv);
					D3DXVec3TransformCoord(&pEdgeVertices[i * 2 + 1], &v, &m_arrBones[nBoneIndex2].matWorldAnim);
				}
				else
				{
					pEdgeVertices[i * 2 + 1] = v;
				}
			}
			hr = m_pEdgeBuffer->Unlock();
		}

		if (m_pBoneBuffer)
		{
			D3DXVECTOR3 *pBoneVertices = NULL;
			hr = m_pBoneBuffer->Lock(0, 0, (BYTE **) &pBoneVertices, D3DLOCK_DISCARD);
			if (SUCCEEDED(hr))
			{
				for (i = 0; i < m_arrBones.size(); i++)
				{
					pBoneVertices[i * 2 + 0].x = m_arrBones[i].matWorldAnim._41;
					pBoneVertices[i * 2 + 0].y = m_arrBones[i].matWorldAnim._42;
					pBoneVertices[i * 2 + 0].z = m_arrBones[i].matWorldAnim._43;
					int nParentIndex = m_arrBones[i].nParentBone;
					if (nParentIndex == -1)
					{
						pBoneVertices[i * 2 + 1].x = 0.0f;
						pBoneVertices[i * 2 + 1].y = 0.0f;
						pBoneVertices[i * 2 + 1].z = 0.0f;
					}
					else
					{
						pBoneVertices[i * 2 + 1].x = m_arrBones[nParentIndex].matWorldAnim._41;
						pBoneVertices[i * 2 + 1].y = m_arrBones[nParentIndex].matWorldAnim._42;
						pBoneVertices[i * 2 + 1].z = m_arrBones[nParentIndex].matWorldAnim._43;
					}
				}

				hr = m_pBoneBuffer->Unlock();
			}
		}

		return hr;
	}

	return E_FAIL;
}

VOID CD3DSkinnedMesh::ComputeBoundingSphere(D3DXVECTOR3 *pCenter, FLOAT *pRadius)
{
	HRESULT	hr = D3DXComputeBoundingSphere(
			&m_arrVertices[0].p,
			(DWORD) m_arrVertices.size(),
			D3DXGetFVFVertexSize(D3DFVF_VERTEX),
			pCenter,
			pRadius);
}

HRESULT CD3DSkinnedMesh::Destroy()
{
	m_pMS3DFile = NULL;
	m_pDevice = NULL;
	if (m_pVertexBuffer)
	{
		m_pVertexBuffer->Release();
		m_pVertexBuffer = NULL;
	}

	if (m_pEdgeBuffer)
	{
		m_pEdgeBuffer->Release();
		m_pEdgeBuffer = NULL;
	}

	if (m_pBoneBuffer)
	{
		m_pBoneBuffer->Release();
		m_pBoneBuffer = NULL;
	}

	m_arrVertices.clear();
	m_arrVertexBoneIds.clear();
	m_arrFlatNormals.clear();
	m_arrSubsets.clear();
	for (DWORD i = 0; i < m_arrShaders.size(); i++)
	{
		if (m_arrShaders[i].pTexture)
			m_arrShaders[i].pTexture->Release();
	}
	m_arrShaders.clear();

	m_bDefaultShaderOnly = FALSE;
	m_bHasFlatNormals = FALSE;

	m_fTime = -1.0f;
	m_fLowestTime = 0.0f;
	m_arrBones.clear();

	return S_OK;
}

DWORD CD3DSkinnedMesh::GetNumSubsets()
{
	return (DWORD) m_arrSubsets.size();
}

HRESULT CD3DSkinnedMesh::SetSubsetShader(DWORD dwSubset)
{
	if (m_pDevice)
	{
		DWORD dwShader = 0; // default shader
		if (!m_bDefaultShaderOnly)
		{
			if (dwSubset >= 0 && dwSubset < m_arrSubsets.size())
				dwShader = m_arrSubsets[dwSubset].AttribId;
		}
		
		D3DMATERIAL8 *mat = &m_arrShaders[dwShader].mat;
		m_pDevice->SetMaterial(mat);
		m_pDevice->SetTexture(0, m_arrShaders[dwShader].pTexture);

		

		m_pDevice->SetRenderState(D3DRS_SPECULARENABLE, TRUE);
		m_pDevice->SetRenderState(D3DRS_DIFFUSEMATERIALSOURCE, D3DMCS_MATERIAL);
		m_pDevice->SetRenderState(D3DRS_SPECULARMATERIALSOURCE, D3DMCS_MATERIAL);
		m_pDevice->SetRenderState(D3DRS_AMBIENTMATERIALSOURCE, D3DMCS_MATERIAL);
		m_pDevice->SetRenderState(D3DRS_EMISSIVEMATERIALSOURCE, D3DMCS_MATERIAL);
		if (m_arrShaders[dwShader].fTransparency < 1.0f)
		{
			m_pDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, TRUE);
			m_pDevice->SetRenderState(D3DRS_SRCBLEND, D3DBLEND_SRCALPHA);
			m_pDevice->SetRenderState(D3DRS_DESTBLEND, D3DBLEND_INVSRCALPHA);
		}
		else
		{
			m_pDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, FALSE);
		}

		m_pDevice->SetTextureStageState(0, D3DTSS_COLORARG1, D3DTA_TEXTURE);
		m_pDevice->SetTextureStageState(0, D3DTSS_COLORARG2, D3DTA_DIFFUSE);
		m_pDevice->SetTextureStageState(0, D3DTSS_COLOROP, D3DTOP_MODULATE);

		m_pDevice->SetTextureStageState(0, D3DTSS_ALPHAARG1, D3DTA_TEXTURE);
		m_pDevice->SetTextureStageState(0, D3DTSS_ALPHAARG2, D3DTA_DIFFUSE);
		m_pDevice->SetTextureStageState(0, D3DTSS_ALPHAOP, D3DTOP_MODULATE);

		if (m_arrShaders[dwShader].pTexture != NULL)
		{
			if (m_arrShaders[dwShader].bSphereMap)
			{
				m_pDevice->SetTextureStageState(0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_CAMERASPACENORMAL);
				m_pDevice->SetTextureStageState(0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2);
			}
			else
			{
				m_pDevice->SetTextureStageState(0, D3DTSS_TEXCOORDINDEX, D3DTSS_TCI_PASSTHRU);
				m_pDevice->SetTextureStageState(0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_DISABLE);
			}
		}

		return S_OK;
	}

	return E_FAIL;
}

HRESULT CD3DSkinnedMesh::DrawSubset(DWORD dwSubset)
{
	if (m_pDevice && dwSubset >= 0 && dwSubset < m_arrSubsets.size())
	{
		m_pDevice->DrawPrimitive(D3DPT_TRIANGLELIST, m_arrSubsets[dwSubset].VertexStart, m_arrSubsets[dwSubset].VertexCount / 3);
	}

	return E_FAIL;
}

HRESULT CD3DSkinnedMesh::Draw()
{
	if (m_pDevice && m_pVertexBuffer)
	{
		m_pDevice->SetVertexShader(D3DFVF_VERTEX);
		m_pDevice->SetStreamSource(0, m_pVertexBuffer, D3DXGetFVFVertexSize(D3DFVF_VERTEX));

		for (DWORD i = 0; i < GetNumSubsets(); i++)
		{
			SetSubsetShader(i);
			DrawSubset(i);
		}
		return S_OK;
	}

	return E_FAIL;
}

HRESULT CD3DSkinnedMesh::DrawEdges()
{
	if (m_pDevice && m_pEdgeBuffer)
	{
		m_pDevice->SetVertexShader(D3DFVF_XYZ);
		m_pDevice->SetStreamSource(0, m_pEdgeBuffer, D3DXGetFVFVertexSize(D3DFVF_XYZ));
		m_pDevice->DrawPrimitive(D3DPT_LINELIST, 0, m_pMS3DFile->GetNumEdges());
		return S_OK;
	}

	return E_FAIL;
}

HRESULT CD3DSkinnedMesh::DrawBones()
{
	if (m_pDevice && m_pBoneBuffer)
	{
		m_pDevice->SetVertexShader(D3DFVF_XYZ);
		m_pDevice->SetStreamSource(0, m_pBoneBuffer, D3DXGetFVFVertexSize(D3DFVF_XYZ));
		m_pDevice->DrawPrimitive(D3DPT_LINELIST, 0, m_pMS3DFile->GetNumJoints());
		return S_OK;
	}

	return E_FAIL;
}

VOID CD3DSkinnedMesh::SetFlatNormals(BOOL bEnable)
{
	if (m_bHasFlatNormals != bEnable)
	{
		m_bHasFlatNormals = bEnable;
		UpdateVertexBuffer();
	}
}

BOOL CD3DSkinnedMesh::HasFlatNormals()
{
	return m_bHasFlatNormals;
}

VOID CD3DSkinnedMesh::SetDefaultShaderOnly(BOOL bEnable)
{
	m_bDefaultShaderOnly = bEnable;
}

BOOL CD3DSkinnedMesh::HasDefaultShaderOnly()
{
	return m_bDefaultShaderOnly;
}

VOID CD3DSkinnedMesh::SetTime(FLOAT fTime)
{
	if (fTime >= 0)
	{
		if (fabs(fTime - m_fTime) < 0.04f)
			return;
	}

	m_fTime = fTime;

	DWORD i;
	if (fTime < 0.0f)
	{
		for (i = 0; i < m_arrBones.size(); i++)
			m_arrBones[i].matWorldAnim = m_arrBones[i].matWorld;
	}
	else
	{
		for (i = 0; i < m_arrBones.size(); i++)
		{
			m_arrBones[i].matObjectAnim = m_arrBones[i].matObject;
//FIND
//			if (m_arrBones[i].pInterpolator)
//			{
			int transkeys = m_arrBones[i].NumTranslationKeys;
			if (transkeys > 0) {
				D3DXVECTOR3 vScale;
				D3DXQUATERNION qRotate;
				D3DXVECTOR3 vTranslate;
				//HRESULT hr;
				//hr = D3D_OK; 
				D3DXVECTOR3 vRotateTest;
				D3DXVECTOR3 vTranslate1;
				D3DXVECTOR3 vTranslate2;
				//Find the right time to intepolate between.
				//Loop through all frames
				unsigned int k;
				unsigned int keys = m_arrBones[i].NumRotationKeys;

				for (k = 1; k < keys; k++)
				{
					float time1 = m_arrBones[i].TranslationKeys[k-1].Time;
					float time2 = m_arrBones[i].TranslationKeys[k].Time;
					if (fTime >= time1  && fTime <= time2)
					{
						//Get the two frames, to interpolate between
						
						float timeDelta = m_arrBones[i].TranslationKeys[k].Time - m_arrBones[i].TranslationKeys[k-1].Time;
						float interpValue = ( float )(( fTime - m_arrBones[i].TranslationKeys[k-1].Time )/timeDelta );
						vTranslate1 = m_arrBones[i].TranslationKeys[k-1].Value;
						vTranslate2 = m_arrBones[i].TranslationKeys[k].Value;
						
						//vTranslate = vTranslate1 + (vTranslate2-vTranslate1) * interpValue;

						//vTranslate = m_arrBones[i].TranslationKeys[k].Value;
						//qRotate = m_arrBones[i].RotationKeys[k].Value;
						D3DXVec3Lerp(&vTranslate,&vTranslate1,&vTranslate2,interpValue);
						D3DXQuaternionSlerp(&qRotate, &m_arrBones[i].RotationKeys[k-1].Value, &m_arrBones[i].RotationKeys[k].Value,interpValue);
					
						D3DXMATRIX matAnim;
						D3DXMatrixRotationQuaternion(&matAnim, &qRotate);

						matAnim._41 = vTranslate.x;
						matAnim._42 = vTranslate.y;
						matAnim._43 = vTranslate.z;

						m_arrBones[i].matObjectAnim = matAnim;
						
						break;
					}

				}
			}
		}

		for (i = 0; i < m_arrBones.size(); i++)
		{
			int nParentJoint = m_arrBones[i].nParentBone;
			if (nParentJoint == -1)
				m_arrBones[i].matWorldAnim = m_arrBones[i].matObjectAnim;
			else
				D3DXMatrixMultiply(&m_arrBones[i].matWorldAnim, &m_arrBones[i].matObjectAnim, &m_arrBones[nParentJoint].matWorldAnim);
		}
	}

	UpdateVertexBuffer();
}

FLOAT CD3DSkinnedMesh::GetTime()
{
	return m_fTime;
}

FLOAT CD3DSkinnedMesh::GetLowestTime()
{
	return m_fLowestTime;
}

FLOAT CD3DSkinnedMesh::GetAnimationFPS()
{
	if (m_pMS3DFile)
		return m_pMS3DFile->GetAnimationFPS();

	return 30.0;
}

FLOAT CD3DSkinnedMesh::GetTotalFrames()
{
	if (m_pMS3DFile)
		return m_pMS3DFile->GetTotalFrames();

	return 0.0;
}

void CD3DSkinnedMesh::SetupBones()
{
	UINT i, j;

	// from MS3D to D3D
	m_arrBones.resize(m_pMS3DFile->GetNumJoints());
	for (i = 0; i < m_pMS3DFile->GetNumJoints(); i++)
	{
		ms3d_joint_t *pJoint = NULL;
		m_pMS3DFile->GetJointAt(i, &pJoint);

		D3DXMatrixFromRotXYZPos(&m_arrBones[i].matObject, pJoint->rotation, pJoint->position);
		int nParentJoint = m_pMS3DFile->FindJointByName(pJoint->parentName);
		if (nParentJoint == -1)
			m_arrBones[i].matWorld = m_arrBones[i].matObject;
		else
			D3DXMatrixMultiply(&m_arrBones[i].matWorld, &m_arrBones[i].matObject, &m_arrBones[nParentJoint].matWorld);
		m_arrBones[i].nParentBone = nParentJoint;
	}

	// inverse
	for (i = 0; i < m_arrBones.size(); i++)
	{
		D3DXMATRIX *pMat = D3DXMatrixInverse(&m_arrBones[i].matWorldInv, NULL, &m_arrBones[i].matWorld);
	}

	if (m_pMS3DFile->GetNumJoints() > 0)
		m_fLowestTime = FLT_MAX;
	else
		m_fLowestTime = 0;

	float fHighestTime = 0.0f;
	for (i = 0; i < m_pMS3DFile->GetNumJoints(); i++)
	{
		ms3d_joint_t *pJoint = NULL;
		m_pMS3DFile->GetJointAt(i, &pJoint);

//		_ASSERTE(pJoint->numKeyFramesRot == pJoint->numKeyFramesTrans);

		UINT nNumKeyFrames = pJoint->numKeyFramesRot;

		std::vector<D3DXKEY_QUATERNION> RotationKeys;
		RotationKeys.resize(nNumKeyFrames);

		std::vector<D3DXKEY_VECTOR3> TranslateKeys;
		TranslateKeys.resize(nNumKeyFrames);

		for (j = 0; j < nNumKeyFrames; j++)
		{
			D3DXMATRIX matAnim, matAnimated;
			D3DXMatrixFromRotXYZPos(&matAnim, pJoint->keyFramesRot[j].rotation, pJoint->keyFramesTrans[j].position);
			D3DXMatrixMultiply(&matAnimated, &matAnim, &m_arrBones[i].matObject);

			float fTime = pJoint->keyFramesRot[j].time * m_pMS3DFile->GetAnimationFPS();
			RotationKeys[j].Time = TranslateKeys[j].Time = fTime;
			if (fTime < m_fLowestTime)
				m_fLowestTime = fTime;

			D3DXQUATERNION q;
			//D3DXQuaternionRotationMatrix(&q, &matAnimated);
			//D3DXQuaternionInverse(&RotationKeys[j].Value, &q);

			D3DXQuaternionRotationMatrix(&RotationKeys[j].Value, &matAnimated);


			TranslateKeys[j].Value.x = matAnimated._41;
			TranslateKeys[j].Value.y = matAnimated._42;
			TranslateKeys[j].Value.z = matAnimated._43;
		}

		//Make space
		m_arrBones[i].RotationKeys = new D3DXKEY_QUATERNION[nNumKeyFrames];
		m_arrBones[i].TranslationKeys = new D3DXKEY_VECTOR3[nNumKeyFrames];		
		m_arrBones[i].NumRotationKeys = nNumKeyFrames;
		m_arrBones[i].NumTranslationKeys = nNumKeyFrames;

		unsigned int k;
		for (k=0; k< nNumKeyFrames;k++) {
			//Copy:
			m_arrBones[i].RotationKeys[k] = RotationKeys[k];
			m_arrBones[i].TranslationKeys[k] = TranslateKeys[k];
		}

	}
}

void CD3DSkinnedMesh::GetVectorAt(int nIndex, D3DXVECTOR3 **ppVector)
{
	if (nIndex >= 0 && nIndex < (int) m_arrVertices.size())
		*ppVector = &m_arrVertices[nIndex].p;
}